//
// (C) Copyright 1999 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
//
// DESCRIPTION:
// Dockable pane class implementation
//
// PURPOSE:
// This class, derived from CDialogBar is responsible for creating
// a dockable pane with an extra splitter to allow the user to size
// the pane when it is docked.
//
#include "stdafx.h"
#include "resourcehelper.h"   // temporary resource class

#include "MyPropertySheet.h"

#include "DockPane.h"
#include "resource.h"
#include "rxmfcapi.h"         // ACAD MFC stuff

/****************************************************************************/
/* ACAD INCLUDES */
/****************************************************************************/
#include "adslib.h"
#include "aced.h"
#include "dbmain.h"
#include "acdbabb.h"
#include "adeskabb.h"
#include "rxregsvc.h"
#include "acgi.h"
#include "acdocman.h"
#include "acedinpt.h"
#include "dbapserv.h"

/****************************************************************************/
/* MAP API INCLUDES */
/****************************************************************************/
#include <MapArxApi.h>
#include <MapProj.h>
#include <MapQuery.h>
#include <MapODDefinition.h>
#include <MapODColumn.h>
#include <MapValue.h>
#include <MapODTable.h>
#include <MapODRecord.h>
#include <MapODIterator.h>
#include <MapBoundary.h>

#include "MapConstants.h"
/****************************************************************************/

#include "MyMapReactors.h"
#include "util.h"

#ifdef _DEBUG
//#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// A custom message to work around a problem getting the miniframe pointer
#define WM_DOCKSTATE_CHANGED        (WM_USER + 17)
#define ID_MAP_TIMER 150

// Minimum size (in pixels) of the dockable pane when floating
static const int kMinFloatingHeight = 10;
static const int kMinFloatingWidth = 60;

//////////////////////////////////////////////////////////////////////////////
// CDockPaneWindowHook
//
// The purpose of this class is to hook the floating mini frame so we can
// intercept the AutoCAD custom message WM_ACAD_KEEPFOCUS
IMPLEMENT_DYNAMIC(CDockPaneWindowHook, CSubclassWnd);

LRESULT CDockPaneWindowHook::WindowProc(UINT msg, WPARAM wp, LPARAM lp)
{
	switch(msg)
	{
	    case WM_ACAD_KEEPFOCUS:
            return TRUE;
        default:break;
    }
	return CSubclassWnd::WindowProc(msg, wp, lp);
}

//////////////////////////////////////////////////////////////////////////////
// CDockPane

IMPLEMENT_DYNAMIC(CDockPane, CAdUiDockControlBar)

CDockPane::CDockPane()
{
	m_bIsMapBusy = FALSE;
}

CDockPane::~CDockPane()
{
	if (m_MiniFrameHook.IsHooked())
	{
		// Unhook the miniframe window
		m_MiniFrameHook.HookWindow(NULL);
	}

}

BOOL CDockPane::Create(CWnd *pParentWnd)
{
	// For now, the toolID will be hard-coded, until Alain gives us a unique value
	// AutoCAD uses this ID to retrieve the location of the pane in the registry
	UINT nControlBarID = AFX_IDW_CONTROLBAR_FIRST + 5;

	// Create the dockable pane window
	CString sWndClass;
	sWndClass = AfxRegisterWndClass(CS_DBLCLKS);
	CRect rect(0, 0, 250, 200);
	if (!CAdUiDockControlBar::Create(sWndClass,
		NULL,
		WS_VISIBLE|WS_CHILD|WS_CLIPCHILDREN,
		rect,
		pParentWnd, nControlBarID))
	{
		return FALSE;
	}

	// Set the window title
	CString sTitle;
	VERIFY(sTitle.LoadString(IDS_DOCKTITLE));
	SetWindowText(sTitle);
	
	// Move the dockable pane window to the last known location
	EnableDocking(CBRS_ALIGN_LEFT|CBRS_ALIGN_RIGHT);
	RestoreControlBar();

	return TRUE;
}

int CDockPane::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CAdUiDockControlBar::OnCreate(lpCreateStruct) == -1)
	{
		return -1;
	}
	
	const char *pszContentClass =
		AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW,
		LoadCursor(NULL, IDC_ARROW));

	
	CTemporaryResourceOverride   useThisDllResource;

	// Instanciate the property sheet
	m_pPropSheet = new CMyPropertySheet(this);	

	DWORD dwStyle = WS_CHILD | WS_VISIBLE | WS_BORDER | WS_DLGFRAME;
	dwStyle &= ~WS_CAPTION;
	CRect rect(0, 0, 0, 0);
	
	if (!m_pPropSheet->Create(this, dwStyle, 0))
	{
		delete m_pPropSheet;
		m_pPropSheet = NULL;
		return 0;
	}

	// initialize all of the property sheet pages
	CPropertyPage* curPage = m_pPropSheet->GetActivePage();
	for( int i = 0; i < 2; i++ )
		m_pPropSheet->SetActivePage(i);
	
	m_pPropSheet->SetActivePage(curPage);
    
	// initialize local tree control pointer array
	m_pTree[0] = &m_pPropSheet->m_Page1.m_tree; //Query Tree Control tab
	m_pTree[1] = &m_pPropSheet->m_Page2.m_tree; //Error Tree Control tab

	//set back pointers to this for communication
	m_pTree[0]->SetParent(this);
	m_pTree[1]->SetParent(this);
		
	//Get Map, Start reactors and populate the tree with query info
	if(!StartReactors())
		return -1;

	InitializeTreeControls();
	UpdateMapInfo();

	return 0;
}

void CDockPane::SizeChanged(CRect *lpRect, BOOL bFloating, int flags)
{
	// size the view
	lpRect->InflateRect(-6, -6);
	
	// size the property sheet
	m_pPropSheet->SetWindowPos(NULL,
			lpRect->left, lpRect->top,
			lpRect->Width(), lpRect->Height(),
			SWP_NOACTIVATE | SWP_NOZORDER);

	CRect rSheet;
	m_pPropSheet->GetWindowRect( &rSheet );
	rSheet.InflateRect(-6, -6);
	
	// size the tab control
	m_pPropSheet->GetTabControl()->SetWindowPos(NULL, 5, 5, rSheet.Width()-10,
		rSheet.Height()-5, SWP_NOZORDER | SWP_NOACTIVATE );
	
	// size the property page tree controls
	for( int i = 0; i < 2; i++ )
	{
		m_pTree[i]->SetWindowPos(NULL, 0, 0, rSheet.Width()-18,
			rSheet.Height()-35, SWP_NOZORDER | SWP_NOACTIVATE);
	}

	// reselect the active page to repaint everything
	m_pPropSheet->SetActivePage( m_pPropSheet->GetActivePage() );
}

BEGIN_MESSAGE_MAP(CDockPane, CAdUiDockControlBar)
    //{{AFX_MSG_MAP(CDockPane)
    ON_WM_CREATE()
	ON_MESSAGE(WM_IDLEUPDATECMDUI, OnIdleUpdateCmdUI)
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
	ON_MESSAGE(WM_DOCKSTATE_CHANGED, OnChangedDockedState)
	ON_MESSAGE(WM_REACTOR_MSG, OnReactorMsg)
END_MESSAGE_MAP()


void CDockPane::ToggleBarState(BOOL bShrinkIfDocked /* = TRUE */)
{
}

BOOL CDockPane::IsWindowVisible() const
{
    if (!CAdUiDockControlBar::IsWindowVisible())
        return FALSE;

    return TRUE;
}

void CDockPane::HookMiniFrame()
{
    CWnd* pMiniFrameWnd = GetParentFrame();
    m_MiniFrameHook.HookWindow(pMiniFrameWnd);
}

LONG CDockPane::OnChangedDockedState(UINT, LONG)
{
    //if we are not floating, unhook the window procedure, otherwise hook it
    if(m_bFloating)
        HookMiniFrame();
    return 0L;
}

LRESULT CDockPane::OnIdleUpdateCmdUI(WPARAM wParam, LPARAM) 
{
	return 0L;
}

void CDockPane::OnTimer(UINT nIDEvent) 
{
	if(!MapIsBusy())
	{
		m_bIsMapBusy = FALSE;
		m_bTimerRunning = FALSE;
		KillTimer(ID_MAP_TIMER);
		UpdateMapInfo();
	}
	
	CAdUiDockControlBar::OnTimer(nIDEvent);
}

LONG CDockPane::OnReactorMsg(UINT lParam, LONG wParam)
{
    TRACE0("OnReactorMsg called\n");

	//check to see if map is dancing with ade
	if(MapIsBusy())
	{
		m_bIsMapBusy = TRUE;
		m_bTimerRunning = TRUE;
		SetTimer(ID_MAP_TIMER, 500, NULL);
		return 0l;
	}
	else
	{
		if(m_bTimerRunning)
		{
			KillTimer(ID_MAP_TIMER);
			m_bTimerRunning = FALSE;;
		}

		UpdateMapInfo();
		return 0l;
	}
	
    return 0l;
}

void CDockPane::OnBarStyleChange(DWORD oldStyle, DWORD newStyle)
{
    m_bFloating = (newStyle & CBRS_FLOATING);
    //This is lame - when this notification comes we can't get a pointer
    //to the miniframe even though its already been created, so we'll
    //post a message to ourselves to swtich it.
    PostMessage(WM_DOCKSTATE_CHANGED);
}

Adesk::Boolean CDockPane::OnLoadDwg()
{
	UpdateMapInfo();
	return Adesk::kTrue;
}

Adesk::Boolean CDockPane::OnUnloadDwg()
{
	return Adesk::kTrue;
}

void CDockPane::SetFloatingSize(const CSize& size)
{
	// TODO
}

void CDockPane::OnSaveComplete(AcDbDatabase *pDwg, const char *pActualName)
{
		// TODO
}

void CDockPane::PostReactorMessage(WPARAM wParam, LPARAM lParam)
{
		// TODO
}

/////////////////////////////////////////////////////////////////////////////
BOOL CDockPane::StartReactors()
{
	BOOL bRet = FALSE;

	AcMapSession *pMapApi = AcMapGetSession();
	ASSERT(NULL != pMapApi);
	if(pMapApi)
	{
		AcMapProject *pProj;
		if (pMapApi->GetProject(pProj))
		{
			m_pDSetReact = new AcMyDrawingSetReactor();
			ASSERT(NULL != m_pDSetReact);
			if(NULL == m_pDSetReact)
				return FALSE;

			m_pDSetReact->SetDlg(this);

			m_pQueryLibraryReact = new AcMyQueryLibraryReactor();
			ASSERT(NULL != m_pQueryLibraryReact);
			if(NULL == m_pQueryLibraryReact)
				return FALSE;

			m_pQueryLibraryReact->SetDlg(this);

			//add all the reactors
			AcMapQueryLibrary *pQLib = NULL ;
			if (pProj->GetQueryLibrary(pQLib))
			{
				pQLib->AddReactor(m_pQueryLibraryReact) ;
			}
			
			AcMapDrawingSet *pDSet = NULL;
			if (pProj->GetDrawingSet(pDSet))
			{
				pDSet->AddReactor(m_pDSetReact);
				bRet = TRUE;
			}

		}
	}
	else
	{
		TRACE0("Can't get Map\n");
		acutPrintf ("\nCan't get Map\n");
	}

	return bRet;
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::InitializeTreeControls()
{
	// create the image list for the tree control
	// and set the tree style using defaults

	CTemporaryResourceOverride   useThisDllResource;
	
	//Query Tree
	m_pTree[0]->CreateImageList( IDB_QRY_TREE_IMAGES, 16, RGB( 0, 128, 128 ));
	m_pTree[0]->SetTreeStyle();

	//Error Tree
	m_pTree[1]->CreateImageList( IDB_ERROR_TREE_IMAGES, 16, RGB( 0, 128, 128 ));
	m_pTree[1]->SetTreeStyle();

	// drag operations
	m_pTree[0]->EnableDragOperations( TRUE );
	m_pTree[1]->EnableDragOperations( FALSE);
}


/////////////////////////////////////////////////////////////////////////////
void CDockPane::UpdateMapInfo()
{
	PopulateQueryInfo();
	PopulateErrorInfo();
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::PopulateQueryInfo()
{
	AcMapSession *mapApi = AcMapGetSession();
	if (mapApi != NULL) 
	{
		AcMapQueryLibrary *pQLib = NULL;
		AcMapProject *pProj;
		if (mapApi->GetProject(pProj) == Adesk::kTrue &&
			pProj->GetQueryLibrary(pQLib))
		{
			HTREEITEM htiParent; // parent item
									
			// add the parent item, make it bold
			m_pTree[0]->DeleteAllItems();
			CString strParent(_T("Query Library"));
			htiParent = m_pTree[0]->AddTreeItem( NULL, strParent, TVI_ROOT, 0, FALSE );
			m_pTree[0]->SetItemState( htiParent, TVIS_BOLD, TVIS_BOLD );

			//add the current query
			PopulateCurrentQueryInfo(pProj, htiParent);

			//add all the categories
			PopulateCategoryQueryInfo(pQLib, htiParent);
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::PopulateErrorInfo()
{
	AcMapSession *mapApi = AcMapGetSession();
	if (mapApi != NULL) 
	{
		AcMapProject *pProj;
		if (mapApi->GetProject(pProj) == Adesk::kTrue)
		{
			AcMapErrorStack *pErr;

			if (mapApi->GetErrorStack(pErr) == Adesk::kFalse)
			{
				acutPrintf ("\nCan't get error stack.\n") ;
				return;
			}
			
			HTREEITEM htiParent; // parent item
			HTREEITEM htiChild;  // child item

			// add the parent item, make it bold
			m_pTree[1]->DeleteAllItems();
			CString strParent(_T("Map Errors"));
			htiParent = m_pTree[1]->AddTreeItem( NULL, strParent, TVI_ROOT, 0, FALSE );
			m_pTree[1]->SetItemState( htiParent, TVIS_BOLD, TVIS_BOLD );

			AcMapErrorEntry *pEntry = NULL ;
			for (int i = 0; i < pErr->CountEntries(); i++)
			{
				CString strMsg;
				if (pErr->GetEntry(pEntry, i))
				{
					char *msg = NULL;
					pEntry->GetErrorMessage(msg);
					strMsg.Format("Error %d: %s", pEntry->ErrorCode(), msg);
					htiChild = m_pTree[1]->AddTreeItem( htiParent, strMsg, TVI_LAST, 1, FALSE );
					delete [] msg;

					AcMapErrorParameter *pPar = NULL ;
					for (int k = 0; k < pEntry->CountParameters(); k++)
					{
						CString strErrorParm;
						if (pEntry->GetParameter(pPar, k))
						{
							char *pP = NULL,
								 *pSQL = NULL;
							pPar->GetParameterDescription(pP);
							pPar->GetStatement(pSQL);
							if (pSQL)
							{
								strErrorParm.Format("Parameter: %s = %s pos:%d", pP, pSQL, pPar->GetErrorPosition());
								m_pTree[1]->AddTreeItem( htiChild, strErrorParm, TVI_LAST, 2, FALSE);
							}
							else
							{
								strErrorParm.Format("Parameter: %s", pP);
								m_pTree[1]->AddTreeItem( htiChild, strErrorParm, TVI_LAST, 2, FALSE);
							}
							delete [] pP;
							delete [] pSQL;
						}

					}
					delete pPar;
				}
			}
			delete pEntry;
			pErr->Clear();
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::PopulateCurrentQueryInfo(AcMapProject *pProj, HTREEITEM htiParent)
{
	AcMap::EErrCode ecode;
	HTREEITEM htiChild;
	CString str;
	AcMapQuery *pQuery = NULL;
	pQuery = GetQuery(TRUE);
	if(pQuery)
	{
		//add the current query to tree
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "Current Query", TVI_LAST, 3, FALSE);
		
		//see if we have any branches defined
		AcMapQueryBranch *pBranch = NULL;
		ecode = pQuery->GetBranch(pBranch);
		if(ecode == AcMap::kOk)
		{
			//get the operands
			int iOperandCount = pBranch->CountOperands();
			if(iOperandCount > 0)
			{
				AcMapQueryUnit *pQueryUnit = NULL;
				for(int i=0; i < iOperandCount; i++)
				{
					ecode = pBranch->GetOperand(pQueryUnit, i);
					if(ecode == AcMap::kOk)
					{
						PopulateQueryBranchInfo(pQueryUnit, htiChild);
					}
				}
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::PopulateQueryBranchInfo(AcMapQueryUnit *pQueryUnit, HTREEITEM htiParent)
{
	AcMap::EClassId Id;

	Id = pQueryUnit->IsA();
		
	//switch the type
	switch(Id)
	{
		case AcMap::kLocationCondition:
		{
			PopulateAcMapLocationCondition(pQueryUnit, htiParent);
			break;
		}
		case AcMap::kPropertyCondition:
		{
			PopulateAcMapPropertyCondition(pQueryUnit, htiParent);
			break;
		}
		case AcMap::kSQLCondition:
		{
			PopulateAcMapSQLCondition(pQueryUnit, htiParent);
			break;
		}
		case AcMap::kDataCondition:
		{
			PopulateAcMapDataCondition(pQueryUnit, htiParent);
			break;
		}
		default:
			break;
	}
}

/////////////////////////////////////////////////////////////////////////////
//AcMapLocationCondition
void CDockPane::PopulateAcMapLocationCondition(AcMapQueryUnit *pQueryUnit, HTREEITEM htiParent)
{
	AcMapLocationCondition *pLocCond = NULL;
	const AcMapLocationBoundary *pLocationBndry = NULL;
	HTREEITEM htiSubChild;  // child item
	Adesk::Boolean bIsNot;
	AcMap::EJoinOperator kOp;
	AcMap::ELocationType kLocationType;
	AcMap::EClassId classId;
	char* pType = NULL;

	//cast it
	pLocCond = (AcMapLocationCondition*)pQueryUnit;
	ASSERT(NULL != pLocCond);
	
	bIsNot = pLocCond->IsNot();
	
	//get the type as a string
	pType = getClassIdString(pLocCond->IsA());
	//get operator
	kOp = pQueryUnit->JoinOperator();
	
	//add the operator type
	if(Adesk::kTrue == bIsNot)
	{
		htiSubChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 6, FALSE);
	}
	else if(kOp == AcMap::kOperatorAnd)
	{
		htiSubChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 4, FALSE);
	}
	else if(kOp == AcMap::kOperatorOr)
	{
		htiSubChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 5, FALSE);
	}

	//add the item
	m_pTree[0]->AddTreeItem(htiSubChild, pType, TVI_LAST, 9, FALSE);

	// Access the location type.
    kLocationType = pLocCond->LocationType();

	// check the type of location query and add the subitem
	char* pLocType = NULL;
	if(kLocationType == AcMap::kLocationInside)
		pLocType = "Location Inside";
	else if(kLocationType == AcMap::kLocationCrossing)
		pLocType = "Location Crossing";

	m_pTree[0]->AddTreeItem(htiSubChild, pLocType, TVI_LAST, 11, FALSE);

	// Access the boundary object.
	pLocationBndry = pLocCond->Boundary();
	ASSERT(pLocationBndry);
	classId = pLocationBndry->IsA();
	switch(classId)
	{
		case AcMap::kAllBoundary:
		{
			PopulateAllBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kPointBoundary:
		{
			PopulatePointBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kCircleBoundary:
		{
			PopulateCircleBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kFenceBoundary:
		{
			PopulateFenceBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kBufferFenceBoundary:
		{
			PopulateBufferFenceBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kPolygonBoundary:
		{
			PopulatePolygonBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kWindowBoundary:
		{
			PopulateWindowBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kPolylineBoundary:
		{
			PopulatePolylineBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kBufferPolylineBoundary:
		{
			PopulateBufferPolylineBoundary(pLocationBndry, htiSubChild);
			break;
		}
		case AcMap::kClosedPolylineBoundary:
		{
			PopulateClosedPolylineBoundary(pLocationBndry, htiSubChild);
			break;
		}
		default:
		{
			ASSERT(0);
			break;
		}
	}
}

void CDockPane::PopulateAllBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	//just insert the item.
	m_pTree[0]->AddTreeItem(htiParent, "All Boundary", TVI_LAST, 17, FALSE);
}
		
void CDockPane::PopulatePointBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapPointBoundary *pPointBoundary = (AcMapPointBoundary*)pLocationBoundary;
	ASSERT(pPointBoundary);

	HTREEITEM htiChild;
	AcGePoint3d thePoint;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Point Boundary", TVI_LAST, 17, FALSE);

	thePoint = pPointBoundary->GetPoint();
	str.Format("x:%.2f y:%.2f z:%.2f", thePoint.x, thePoint.y, thePoint.z);

	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
}
		
void CDockPane::PopulateCircleBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapCircleBoundary *pCircleBoundary = (AcMapCircleBoundary*)pLocationBoundary;
	ASSERT(pCircleBoundary);

	HTREEITEM htiChild;
	double radius;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Circle Boundary", TVI_LAST, 17, FALSE);

	radius = pCircleBoundary->GetRadius();
	str.Format("Radius:%f", radius);

	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
}
		
void CDockPane::PopulateFenceBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapFenceBoundary *pFenceBoundary = (AcMapFenceBoundary*)pLocationBoundary;
	ASSERT(pFenceBoundary);

	HTREEITEM htiChild;
	AcGePoint3dArray pointArray;
	AcGePoint3d thePoint;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Fence Boundary", TVI_LAST, 17, FALSE);

	pointArray = pFenceBoundary->GetPoints();
	for(int i=0; i < pointArray.length(); i++)
	{
		thePoint = pointArray[i];
		str.Format("x:%.2f y:%.2f z:%.2f", thePoint.x, thePoint.y, thePoint.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}
	
}
		
void CDockPane::PopulateBufferFenceBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapFenceBoundary *pFenceBoundary = (AcMapFenceBoundary*)pLocationBoundary;
	ASSERT(pFenceBoundary);

	HTREEITEM htiChild;
	AcGePoint3dArray pointArray;
	AcGePoint3d thePoint;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "BufferFence Boundary", TVI_LAST, 17, FALSE);

	pointArray = pFenceBoundary->GetPoints();
	for(int i=0; i < pointArray.length(); i++)
	{
		thePoint = pointArray[i];
		str.Format("x:%.2f y:%.2f z:%.2f", thePoint.x, thePoint.y, thePoint.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}
}
		
void CDockPane::PopulatePolygonBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapPolygonBoundary *pPolyBoundary = (AcMapPolygonBoundary*)pLocationBoundary;
	ASSERT(pPolyBoundary);

	HTREEITEM htiChild;
	AcGePoint3dArray pointArray;
	AcGePoint3d thePoint;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Polygon Boundary", TVI_LAST, 17, FALSE);

	pointArray = pPolyBoundary->GetPoints();
	for(int i=0; i < pointArray.length(); i++)
	{
		thePoint = pointArray[i];
		str.Format("x:%.2f y:%.2f z:%.2f", thePoint.x, thePoint.y, thePoint.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}
}
		
void CDockPane::PopulateWindowBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapWindowBoundary *pWindowBoundary = (AcMapWindowBoundary*)pLocationBoundary;
	ASSERT(pWindowBoundary);

	HTREEITEM htiChild;
	AcGePoint3dArray pointArray;
	AcGePoint3d thePoint;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Window Boundary", TVI_LAST, 17, FALSE);

	pointArray = pWindowBoundary->GetPoints();
	for(int i=0; i < pointArray.length(); i++)
	{
		thePoint = pointArray[i];
		str.Format("x:%.2f y:%.2f z:%.2f", thePoint.x, thePoint.y, thePoint.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}
}
		
void CDockPane::PopulatePolylineBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapPolylineBoundary *pPolylineBoundary = (AcMapPolylineBoundary*)pLocationBoundary;
	ASSERT(pPolylineBoundary);

	HTREEITEM htiChild;
	AcMapVertexArray vertexArray;
	AcMapPolylineVertex polyVertex;
	AcGeVector3d normal;
	double dBulge;
	AcGePoint3d vertex;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Window Boundary", TVI_LAST, 17, FALSE);

	vertexArray = pPolylineBoundary->GetPoints();
	for(int i=0; i < vertexArray.Length(); i++)
	{
		polyVertex = vertexArray[i];

		dBulge = polyVertex.GetBulge();
		vertex = polyVertex.GetVertex();
		
		str.Format("Vertex x:%.2f y:%.2f z:%.2f", vertex.x, vertex.y, vertex.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
		str.Format("Bulge:%.f", dBulge);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}

	normal = pPolylineBoundary->GetNormal();
	str.Format("Normal:%.f", normal);
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
    
	if(pPolylineBoundary->IsClosed())
		str = "Closed";
	else
		str = "Not Closed";
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);

}
		
void CDockPane::PopulateBufferPolylineBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapBufferPolylineBoundary *pBufferPolylineBoundary = (AcMapBufferPolylineBoundary*)pLocationBoundary;
	ASSERT(pBufferPolylineBoundary);

	HTREEITEM htiChild;
	AcMapVertexArray vertexArray;
	AcMapPolylineVertex polyVertex;
	AcGeVector3d normal;
	double dBulge;
	double dWidth;
	AcGePoint3d vertex;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Window Boundary", TVI_LAST, 17, FALSE);

	vertexArray = pBufferPolylineBoundary->GetPoints();
	for(int i=0; i < vertexArray.Length(); i++)
	{
		polyVertex = vertexArray[i];

		dBulge = polyVertex.GetBulge();
		vertex = polyVertex.GetVertex();
		
		str.Format("Vertex x:%.2f y:%.2f z:%.2f", vertex.x, vertex.y, vertex.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
		str.Format("Bulge:%.f", dBulge);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}

	dWidth = pBufferPolylineBoundary->GetWidth();
	str.Format("Width:%.f", dWidth);
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	
	normal = pBufferPolylineBoundary->GetNormal();
	str.Format("Normal:%.f", normal);
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
    
	if(pBufferPolylineBoundary->IsClosed())
		str = "Closed";
	else
		str = "Not Closed";
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
}
		
void CDockPane::PopulateClosedPolylineBoundary(const AcMapLocationBoundary *pLocationBoundary, HTREEITEM htiParent)
{
	AcMapClosedPolylineBoundary *pClosedPolylineBoundary = (AcMapClosedPolylineBoundary*)pLocationBoundary;
	ASSERT(pClosedPolylineBoundary);

	HTREEITEM htiChild;
	AcMapVertexArray vertexArray;
	AcMapPolylineVertex polyVertex;
	AcGeVector3d normal;
	double dBulge;
	AcGePoint3d vertex;
	CString str;
	
	htiChild = m_pTree[0]->AddTreeItem(htiParent, "Window Boundary", TVI_LAST, 17, FALSE);

	vertexArray = pClosedPolylineBoundary->GetPoints();
	for(int i=0; i < vertexArray.Length(); i++)
	{
		polyVertex = vertexArray[i];

		dBulge = polyVertex.GetBulge();
		vertex = polyVertex.GetVertex();
		
		str.Format("Vertex x:%.2f y:%.2f z:%.2f", vertex.x, vertex.y, vertex.z);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
		str.Format("Bulge:%.f", dBulge);
		m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
	}

	normal = pClosedPolylineBoundary->GetNormal();
	str.Format("Normal:%.f", normal);
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
    
	if(pClosedPolylineBoundary->IsClosed())
		str = "Closed";
	else
		str = "Not Closed";
	m_pTree[0]->AddTreeItem(htiChild, str, TVI_LAST, 18, FALSE);
}

/////////////////////////////////////////////////////////////////////////////
//AcMapDataCondition
void CDockPane::PopulateAcMapPropertyCondition(AcMapQueryUnit *pQueryUnit, HTREEITEM htiParent)
{
	AcMapPropertyCondition *pPropCond = NULL;
	HTREEITEM htiChild, htiSubChild;  // child item
	Adesk::Boolean bIsNot;
	AcMap::EJoinOperator kOp;
	AcMap::EConditionOperator kCondition;
	AcMap::EPropertyType kPropType;
	char* pType = NULL;
	char* pPropType = NULL;
	char* pConditionOp = NULL;
	const char * pcValue = NULL;
	
	pPropCond = (AcMapPropertyCondition*)pQueryUnit;
	ASSERT(NULL != pPropCond);

	bIsNot = pPropCond->IsNot();

	//get the type as a string
	pType = getClassIdString(pPropCond->IsA());
	//get operator
	kOp = pQueryUnit->JoinOperator();
	
	//add the operator type
	if(Adesk::kTrue == bIsNot)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 6, FALSE);
	}
	else if(kOp == AcMap::kOperatorAnd)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 4, FALSE);
	}
	else if(kOp == AcMap::kOperatorOr)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 5, FALSE);
	}

	//add the item
	htiSubChild = m_pTree[0]->AddTreeItem(htiChild, pType, TVI_LAST, 10, FALSE);
	
	// Get the property type and add sub item
	kPropType = pPropCond->PropertyType();
	pPropType = getPropertyTypeString(kPropType);
	m_pTree[0]->AddTreeItem(htiSubChild, pPropType, TVI_LAST, 13, FALSE);
	
	// Get the condition operator  and add sub item
	kCondition = pPropCond->ConditionOperator();
	pConditionOp = getPropertyConditionString(kCondition);
	m_pTree[0]->AddTreeItem(htiSubChild, pConditionOp, TVI_LAST, 11, FALSE);
	
	// Get the expression value and add sub item
	pcValue = pPropCond->Value();
	m_pTree[0]->AddTreeItem(htiSubChild, pcValue, TVI_LAST, 12, FALSE);

}

/////////////////////////////////////////////////////////////////////////////
//AcMapPropertyCondition
void CDockPane::PopulateAcMapDataCondition(AcMapQueryUnit *pQueryUnit, HTREEITEM htiParent)
{
	AcMapDataCondition *pDataCond = NULL;
	HTREEITEM htiChild, htiSubChild;  // child item
	Adesk::Boolean bIsNot;
	AcMap::EJoinOperator kOp;
	AcMap::EConditionOperator kCondition;
	AcMap::EDataQueryType kDataType;
	char* pType = NULL;
	char* pDataType = NULL;
	char* pConditionOp = NULL;
	const char* pcValue = NULL;
	const char* pcTableName = NULL;
	const char* pcColumnName = NULL;
		
	pDataCond = (AcMapDataCondition*)pQueryUnit;
	ASSERT(NULL != pDataCond);

	bIsNot = pDataCond->IsNot();

	//get the type as a string
	pType = getClassIdString(pDataCond->IsA());
	//get operator
	kOp = pQueryUnit->JoinOperator();
	
	//add the operator type
	if(Adesk::kTrue == bIsNot)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 6, FALSE);
	}
	else if(kOp == AcMap::kOperatorAnd)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 4, FALSE);
	}
	else if(kOp == AcMap::kOperatorOr)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 5, FALSE);
	}

	//add the item
	htiSubChild = m_pTree[0]->AddTreeItem(htiChild, pType, TVI_LAST, 7, FALSE);
	
    // Get the data type and add sub item
	kDataType = pDataCond->DataType();
	pDataType = getDataTypeString(kDataType);
	m_pTree[0]->AddTreeItem(htiSubChild, pDataType, TVI_LAST, 13, FALSE);

    // Get the table name.
    pcTableName = pDataCond->Table();
	m_pTree[0]->AddTreeItem(htiSubChild, pcTableName, TVI_LAST, 14, FALSE);

    // Get the column name.
	pcColumnName = pDataCond->Column();
    m_pTree[0]->AddTreeItem(htiSubChild, pcColumnName, TVI_LAST, 15, FALSE);

    	
	// Get the condition operator  and add sub item
	kCondition = pDataCond->ConditionOperator();
	pConditionOp = getPropertyConditionString(kCondition);
	m_pTree[0]->AddTreeItem(htiSubChild, pConditionOp, TVI_LAST, 11, FALSE);
	
	// Get the expression value and add sub item
	pcValue = pDataCond->Value();
	m_pTree[0]->AddTreeItem(htiSubChild, pcValue, TVI_LAST, 12, FALSE);
}

/////////////////////////////////////////////////////////////////////////////
//AcMapSQLCondition
void CDockPane::PopulateAcMapSQLCondition(AcMapQueryUnit *pQueryUnit, HTREEITEM htiParent)
{
	AcMapSQLCondition *pSQLCond = NULL;
	HTREEITEM htiChild, htiSubChild;  // child item
	Adesk::Boolean bIsNot;
	AcMap::EJoinOperator kOp;
	char* pType = NULL;
	const char* pcLPN = NULL;
	const char* pcWhereCondition = NULL;
			
	pSQLCond = (AcMapSQLCondition*)pQueryUnit;
	ASSERT(NULL != pSQLCond);

	bIsNot = pSQLCond->IsNot();

	//get the type as a string
	pType = getClassIdString(pSQLCond->IsA());
	//get operator
	kOp = pQueryUnit->JoinOperator();
	
	//add the operator type
	if(Adesk::kTrue == bIsNot)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 6, FALSE);
	}
	else if(kOp == AcMap::kOperatorAnd)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 4, FALSE);
	}
	else if(kOp == AcMap::kOperatorOr)
	{
		htiChild = m_pTree[0]->AddTreeItem(htiParent, "-", TVI_LAST, 5, FALSE);
	}

	//add the item
	htiSubChild = m_pTree[0]->AddTreeItem(htiChild, pType, TVI_LAST, 8, FALSE);
		
	// Get the LPN.
    pcLPN = pSQLCond->LinkTemplate();
	m_pTree[0]->AddTreeItem(htiSubChild, pcLPN, TVI_LAST, 11, FALSE);
    
	// get the WhereCondition
	pcWhereCondition = pSQLCond->WhereCondition();
	m_pTree[0]->AddTreeItem(htiSubChild, pcWhereCondition, TVI_LAST, 11, FALSE);
}

/////////////////////////////////////////////////////////////////////////////
void CDockPane::PopulateCategoryQueryInfo(AcMapQueryLibrary *pQLib, HTREEITEM htiParent)
{
	AcMapQueryCategory *pQCat = NULL;
	HTREEITEM htiChild, htiSubChild; // child item
	AcMapQuery *pQuery;
	const char *pcName;
	const char *pcQryName;
	int iCatQty = pQLib->CountCategories();

	//add all the categories
	for (int i = 0; i < iCatQty; i++)
	{
		if (pQLib->GetCategory(pQCat, i) == AcMap::kOk)
		{
			pQCat->GetName(pcName);
								
			htiChild = m_pTree[0]->AddTreeItem( htiParent, pcName, TVI_LAST, 1, FALSE );
								
			int iQryQty = pQCat->CountQueries();
			AcMapQueryAttribute *pQAttr = NULL;
			for (int j = 0; j < iQryQty; j++)
			{
				if (pQCat->GetQuery(pQAttr, j) == AcMap::kOk)
				{
					pQAttr->GetName(pcQryName);

					//add query to tree
					htiSubChild = m_pTree[0]->AddTreeItem( htiChild, pcQryName, TVI_LAST, 3, FALSE);

					//get a new query object
					pQuery = GetQuery(FALSE);
					AcMap::EErrCode ecode;
					ecode = pQuery->Load(pcName, pcQryName);
					if(ecode == AcMap::kOk)
					{
						AcMapQueryBranch *pBranchObj = NULL;
						if(pQuery->GetBranch(pBranchObj) == AcMap::kOk)
						{
							//get the operands
							int iOperandCount = pBranchObj->CountOperands();
							if(iOperandCount > 0)
							{
								AcMapQueryUnit *pQueryUnit = NULL;
								for(int i=0; i < iOperandCount; i++)
								{
									if(pBranchObj->GetOperand(pQueryUnit, i) == AcMap::kOk)
									{
										PopulateQueryBranchInfo(pQueryUnit, htiSubChild);
									}
								}
							}
						}
					}
					else
					{
						PopulateErrorInfo();
					}
				}
			}
			if(NULL != pQAttr)
				delete pQAttr;
		}
	}

	m_pTree[0]->ExpandBranch(htiParent);
	
	if(NULL != pQCat)
		delete pQCat;
}


//////////////////////////////////////
//Notification from the Map Reactors
//////////////////////////////////////

//////////////////////////////////////
//AcMapDrawingSetReactor
	
// The notification that the drawing is attached to the Drawing Set
/////////////////////////////////////////////////////////////////////////////
void CDockPane::DrawingAttached(const AcMapAttachedDrawing *pObj)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing attached %s\n", pcName) ;
	}

	UpdateMapInfo();
}

// The notification that the drawing is detached from the Drawing Set
/////////////////////////////////////////////////////////////////////////////
void CDockPane::DrawingDetached(unsigned long Id, const char *pcName, int level)
{
	acutPrintf ("\n***NOTIFICATION:Drawing %s detached\n", pcName) ;

	UpdateMapInfo();
}

// The notification that the  drawing is activated
/////////////////////////////////////////////////////////////////////////////
void CDockPane::DrawingActivated(const AcMapAttachedDrawing *pObj)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing activated %s\n", pcName) ;
	}

	UpdateMapInfo();
}

// The notification that the drawing is deactivated
/////////////////////////////////////////////////////////////////////////////
void CDockPane::DrawingDeactivated(const AcMapAttachedDrawing *pObj)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing deactivated %s\n", pcName) ;
	}

	UpdateMapInfo();
}

// The notification that the drawing settings are modified
/////////////////////////////////////////////////////////////////////////////
void CDockPane::DrawingSettingsModified(const AcMapAttachedDrawing *pObj)
{
	const char *pcName ;
	pObj->GetAliasPath(pcName) ;
	if (pcName)
	{
		acutPrintf ("\n***NOTIFICATION:Drawing settings modified %s\n", pcName) ;
	}

	UpdateMapInfo();
}

//////////////////////////////////////
//AcMapAliasesReactor
	
// The notification that the drawing is attached to the Drawing Set
/////////////////////////////////////////////////////////////////////////////
void CDockPane::AliasAdded(const AcMapDriveAlias *pObj)
{
	UpdateMapInfo();
}


// The notification that the drawing is detached from the Drawing Set
/////////////////////////////////////////////////////////////////////////////
void CDockPane::AliasDeleted(const char *, const char *)
{
	UpdateMapInfo();
}

//////////////////////////////////////
//AcMapQueryLibraryReactor
	
// Adding/Removing query category events.
//
void CDockPane::QueryCategoryAdded(const AcMapQueryCategory *pObj)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryCategoryAdded);
}

void CDockPane::QueryCategoryDeleted(AcMapId Id, const char *pcName)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryCategoryDeleted);
}

// Renaming query category event.
//
void CDockPane::QueryCategoryRenamed(const AcMapQueryCategory *pObj)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryCategoryRenamed);
}

// Adding/Removing query events.
//
void CDockPane::QueryAdded(const AcMapQueryAttribute *pObj)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryAdded);
}

void CDockPane::QueryDeleted(AcMapId Id, const char *pcName)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryDeleted);
}

// Modification query events.
//
void CDockPane::QueryRenamed(const AcMapQueryAttribute *pObj)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryRenamed);
}

void CDockPane::QueryModified(const AcMapQueryAttribute *pObj)
{
	PostMessage(WM_REACTOR_MSG, MSG_QueryModified);
}

/////////////////////////////////////////////////////
//GetQuery

AcMapQuery* CDockPane::GetQuery(BOOL bIsCurrent)
{
    AcMapSession *mapApi = NULL;
	AcMapProject *pProj = NULL;
    AcMapQuery *pQuery = NULL;
   
    do {
        mapApi = AcMapGetSession();
        if (mapApi == NULL) 
        {
            acutPrintf ("\nCan't connect ADE\n");
            break;
        }
        
		if (mapApi->GetProject(pProj) == Adesk::kFalse)
		{
		    acutPrintf ("\nCan't get MAP project\n");
            break;
        }
        
		if (pProj->CreateQuery(pQuery, bIsCurrent) != AcMap::kOk)
		{
            acutPrintf ("\nCan't create query object.\n");
            break;
        }

	} while (0);

	return pQuery;
}

BOOL CDockPane::MapIsBusy()
{
	//check to see if map is dancing with ade
	BOOL bRet = FALSE;
	AcMapSession *mapSession = NULL;
	AcMapProject *pProj = NULL;

	try
	{
		mapSession = AcMapGetSession();
		if(mapSession) 
		{
			if(mapSession->GetProject(pProj) == Adesk::kTrue)
			{
				if(pProj->IsMapActive() == Adesk::kTrue)
				{
					bRet = TRUE;
				}
			}
		}
	}
	catch(...)
	{
		acutPrintf ("\nCan't get map status.\n");
	}

	return bRet;
}
